 
;;; file fde.lsp
;;; fde are frequency polygons-histograms density estimators
;;; wfde objects, are the window objects for fde objects
;;; extracted and specialized from kde on june 95

;;; There is a paper on this file, available in http://libiya.upf.es/soft/fde
;;; Frederic Udina, october 1999.


(defvar *fde-directory* (make-pathname
			 :directory (directory-namestring *load-truename*)
			 :device (pathname-device *load-truename*)))

(defun make-histogram (data &rest key-value-pairs)
"Constructs a rich histogram for DATA,
 bin width is computed from normal-ref rule.
 Other args must be key/value pairs to be passed to
 method :isnew for fde-proto:
 bin-width    defaults to normal-ref rule: bw = 3.5 x sigma x n^(-1/3)
 anchor-base can be zero or data-based
 anchor-shift is a [0 1) number to be subtracted after multiplying by bw
              when data-based defaults to 0.5, the first bin edge will be xmin - 0.5 bw
              when zero, default is zero
 x-range      defaults to xmin..xmax expanded by oversmoothed bin-width
 show         whether to show it in a window, defaults to true
 what-to-show What estimates to show, must be a list containing 1+ of
              :histo-lines regular histogram
              :hohistolines hollow histogram
              :fpoly-lines frequency polygon
              :edgpoly-lines edge frequencies polygon
              :linpoly-lines linearly binned freq polygon
              :piecelin-lines piecewise linear estimator
              :boxplot-lines
              :data data points
              :density-lines a (fitted normal) density shadow
              defaults to '(:histo-lines)
 density-num-args the number of args (default 1) for
 density      a function to compute some density function for reference
 Example calls:
 (make-histogram my-data :x-range '(0 5) :show nil)
 (make-histogram (read-data-columns \"myfile.dat\" 1))
 etc."
(apply #'send fde-proto :new :data (coerce data 'vector) key-value-pairs))

;;;
;;; Object fde-proto holds statistical data and methods
;;; to compute histograms, frequency polygons in its
;;; various versions.
;;;


(defproto fde-proto 
  '(data data-summary scale-estimate
	 x-range bin-edges y-scale
	 title
	 bin-width bw-ends
	 bin-counts long-bin-counts half-bin-counts
	 stability-index
	 boxplot-lines 
	 hohisto-lines histo-lines piecelin-lines
	 fpoly-lines edgpoly-lines linpoly-lines
	 anchor-base anchor-shift
	 density density-lines density-num-args
	 info-strings
	 what-to-show
	 window
	 window-up-to-date
	 histos yatracos);these two are out of depend. chain
  '(dependence-chain) *object*
  )

;;;dependence-chain implements the computational flow 
;; using a dependence tree
(send fde-proto :slot-value 'dependence-chain
      '((init-all data anchor-base anchor-shift what-to-show y-scale)
	(data bw-ends data-summary scale-estimate bin-width )
	(bw-ends x-range)
	(data-summary x-range boxplot-lines)
	(y-scale density-lines hohisto-lines histo-lines fpoly-lines 
		    edgpoly-lines piecelin-lines boxplot-lines)
	(bin-width bin-edges)
	(x-range bin-edges density-lines)
	(density density-lines window-up-to-date)
	(anchor-base bin-edges)
	(anchor-shift bin-edges)
	(bin-edges bin-counts half-bin-counts long-bin-counts)
	(bin-counts hohisto-lines histo-lines fpoly-lines 
		    edgpoly-lines piecelin-lines)
	(long-bin-counts  stability-index linpoly-lines)
	(hohisto-lines  window-up-to-date)
	(histo-lines  window-up-to-date)
	(fpoly-lines  window-up-to-date)
	(edgpoly-lines window-up-to-date)
	(piecelin-lines window-up-to-date)
	(linpoly-lines window-up-to-date)
	(density-lines window-up-to-date)
	(what-to-show window-up-to-date)))

(defmeth fde-proto :isnew ( &key (title "Rich histogram")
                               data
			       bin-width
			       anchor-shift anchor-base
			       x-range density
                               (show t)
			       what-to-show
			       (debug nil))
"Initializes a fde-proto descendent.
 The usual way to create a fde-object is the `make-histogram' function."
  (when debug
	(print "fde-proto :isnew got arguments:")
	(print (list   :show show
		       :data data :title title
		       :bin-width    bin-width    
		       :anchor-shift anchor-shift
		       :anchor-base anchor-base
		       :x-range	     x-range      
		       :what-to-show what-to-show 
		       :debug debug)))

  (send self :mark-changed 'init-all)
  (if density (send self :density density)
    (send self :density nil))
  (when title (send self :title title))
  (when data (send self :data (coerce data 'vector)))
  (when bin-width (send self :bin-width bin-width))
  (when anchor-shift (send self :anchor-shift anchor-shift))
  (when x-range (send self :x-range x-range))
  (when what-to-show (send self :what-to-show what-to-show))
  (if (and show (system-has-windows))
      (send self :have-window :show show))
  self)

(defmeth fde-proto :print (&optional (stream t))
   (format stream "#<fde: ~S [~S]>"
	   (send self :title)
	   (call-next-method nil)))


;;;
;;; Accessor methods
;;; all follow the same pattern
;;;
;;;

(defmeth fde-proto :mark-changed (symb)
"Puts 'must-compute on all slots that depend
 on SYMB, according 'dependence-chain"
  (let ((seq (find symb (slot-value 'dependence-chain)
		   :key #'first)))
    (when seq
	  (setf seq (cdr seq))
	  (mapcar #'(lambda (slo)
		      (slot-value slo 'must-compute)
		      (send self :mark-changed slo))
		  seq))))

;;; Standard Accessor methods

#|
;;; all follow the same pattern
(defmeth fde-proto :slot-name (&optional (value nil vset))
  (if vset
      (progn
	;;set all dependants to nil
	(send self :mark-changed 'slot-name)
	(slot-value 'slot-name value))
    ;;not vset, compute if needed
    (when (eq 'must-compute (slot-value 'slot-name))
	  (slot-value 'slot-name
		      (compute-slot-value-default))))
  (slot-value 'slot-name))
|#

;;a couple of macros make it easier to create them:

(defmacro defmeth-fde-proto-accessor-changes (keyword help-string computation)
"Macro Args: keyword help-string computation.
 Defines a fde-proto method KEYWORD to access the slot with same name, providing
 for transmission of changes thorugh the depencence chain. COMPUTATION msut be a body
 to be evaluated when computing the value for the slot."
  (let ((sym (find-symbol (symbol-name keyword))))
    `(defmeth fde-proto 
       ,keyword
       (&optional (value nil vset))
       ,help-string
       (if vset (unless (eql value (slot-value ',sym value))
		  (send self :mark-changed ',sym)
		  (setf (slot-value ',sym) value))
	 (when (eq 'must-compute (slot-value ',sym))
	   (setf (slot-value ',sym) ,computation)))
       (slot-value ',sym))))

(defmacro defmeth-fde-proto-accessor-changes-draw (keyword help-string computation)
"Macro Args: keyword help-string computation.
 Defines a fde-proto method KEYWORD to access the slot with same name, providing
 for transmission of changes thorugh the depencence chain. COMPUTATION msut be a body
 to be evaluated when computing the value for the slot.
 The method will accept a :draw keyword argument. When t, it will redraw the window."
  (let ((sym (find-symbol (symbol-name keyword))))
    `(defmeth fde-proto 
       ,keyword
       (&optional (value nil vset) &key (draw nil))
       ,help-string
       (if vset (progn
		  (send self :mark-changed ',sym)
		(setf (slot-value ',sym) value)
		(when draw (send self :to-window :adjust-to-data)))
       (when (eq 'must-compute (slot-value ',sym))
	 (setf (slot-value ',sym) ,computation)))
     (slot-value ',sym))))




(defmeth-fde-proto-accessor-changes :title "Sets or retrieves the 'title slot"
  (slot-value 'title
	      "Default title never occurs"))

(defmeth-fde-proto-accessor-changes-draw :data 
  "Sets or retrieves the 'data slot.
 If some data is given and :DRAW is t, window is redrawn"
  (error (progn (message-dialog 
		 "There is no data for the histogram")
		"no data")))

(defmeth-fde-proto-accessor-changes-draw :bin-width
			    "Sets or retrieves the bin width for the histograms.
 It defaults to normal-reference with robust estimation for the scale.
 If :DRAW is t, window is redrawn"
			    (send self :compute-default-bin-width :type 'normal-reference))

(defmeth-fde-proto-accessor-changes :data-summary
			    "data-summary is a list of: a list with the five numbers
 and a list with length, mean and stdev"
			    (let ((dt (send self :data)))
			      (list (fivnum dt)
				    (list (length dt)
					  (mean dt)
					  (standard-deviation dt)))))

(defmeth-fde-proto-accessor-changes :scale-estimate
		       "Computes a robust estimate for the scale following 
Janssen, P., Marron, J.S., Veraverbeke, N., and Sarle, W. (1995)"
		       (progn
			 (require "superscale"
				  (merge-pathnames (make-pathname :name "superscale") 
						   *fde-directory*))
			 (if (find "superscale" *modules* :test #'string=)
			     (superscale (send self :data))
			   (car (cddadr (send self :data-summary))))))

(defmeth-fde-proto-accessor-changes-draw :x-range
"Sets or retrieves the 'x-range slot.
 It is the full range of estimation, it is an error if some
 data point falls outside of it.
 If :DRAW is t, the window is redrawn"
(let* ((ds (send self :data-summary))
		  (mbw (second (send self :bw-ends))))
	     (list (- (caar ds) mbw)
		   (+ (nth 4 (car ds)) mbw))))

(defmeth-fde-proto-accessor-changes :bw-ends
;;we take as maximum bin width the maximum of the two oversmoothing values 
;;in Scott's book (1992, Wiley), page 74, and increase it by 20%
"Sets/retrieves slot 'bw-ends, that contain the minimum and maximum
 allowed values for bin width.
 It defaults  to 5% and 120% of the 
 Scott's oversmoothed bin width (scott 19992)"
(let* ((ds (send self :data-summary))
       (iqr (- (fourth (first ds)) (second (first ds))))
       (sig (send self :scale-estimate))
       (n3 (^ (first (second ds)) -0.3333333))
       (os (max (* 2.603 iqr n3) (* 3.729 sig n3))))
  (* os '(0.05 1.2))))

(defmeth-fde-proto-accessor-changes :anchor-base
"Sets/retrieves the 'anchor-base slot. It contains 'data-based or zero
 and is the reference used to shift the histogram anchor 
 using 'anchor-shift"
'data-based)

(defmeth-fde-proto-accessor-changes-draw :y-scale
"Sets/retrieves the 'y-scale slot. It can be 'count 'frequency
 or 'density. Default is density."
'density)

(defmeth fde-proto :y-factor ()
"the factor to multiply counts by to 
 get the y value, given y-scale"
(case (send self :y-scale)
      ('count 1)
      ('frequency (/ (caadr (send self :data-summary))))
      ('density (/ 1 (send self :bin-width) (caadr (send self :data-summary))))))

(defmeth-fde-proto-accessor-changes :bin-edges
"Sets/retrieves 'bin-edges slot, that contains a list of edges
 for the histogram. If needed, it's computed from anchor and bin width."
(let* ((bw (send self :bin-width))
       (xr (send self :x-range))
       (xmin (nth 0 (first (send self :data-summary))))
       (xmax (nth 4 (first (send self :data-summary))))
       b1 nlow nhi blow bhi)
  (cond ((eq (send self :anchor-base) 'data-based)
	 (setf b1 (- xmin (* bw (send self :anchor-shift))))
	 (setf nlow (ceiling (/ (abs (- (min (first xr) xmin)
					b1))
				bw)))
	 (setf blow (- b1 (* bw nlow)))
	 (setf nhi (ceiling (/ (abs (- (max xmax (second xr))
				       b1))
			       bw)))
	 (setf bhi (+ b1 (* bw nhi)))
	 (rseq blow bhi (+ nlow nhi 1)))
	((eq (send self :anchor-base) 'zero)
	 (setf b1 (- (* bw (send self :anchor-shift))))
	 (setf nlow (floor (/ xmin bw)))
	 (setf nhi (ceiling (+ (/  xmax bw) 1)))
	 (setf blow (+ b1 (* nlow bw)))
	 (setf bhi (+ b1 (* nhi bw)))
	 (rseq blow bhi (1+ (- nlow nhi))))
	((and (numberp (send self :anchor-base));;; THIS DOESN'T work<<<<
	      (< (send self :anchor-base) xmin))
	 (setf b1 (- (send self :anchor-base)
		     (* bw (send self :anchor-shift))))
	 (setf nlow (floor (/ xmin bw)))
	 (setf nhi (ceiling (+ (/  xmax bw) 1)))
	 (setf blow (+ b1 (* nlow bw)))
	 (setf bhi (+ b1 (* nhi bw)))
	 (rseq blow bhi (1+ (- nlow nhi))))
	(t (error "anchor-base ~a wrong, must be 'data-based 'zero or a number" 
		  (send self :anchor-base))))))


(defmeth-fde-proto-accessor-changes :bin-counts
"bin-counts are computed. As a side effect, 
also long-bin-counts for the stability index and 
half-bin-counts for the piecelin polygon are computed too"
(let* ((bw (send self :bin-width))
       (xl (first (send self :bin-edges)))
       (bc (make-array (1- (length (send self :bin-edges)))
		       :initial-element 0))
       (lbc (make-array (* 100 (1- (length (send self :bin-edges))))
			:initial-element 0))
       (2bc (make-array (* 2 (1- (length (send self :bin-edges))))
			:initial-element 0))
       (dt (send self :data))
       (ll (length dt))
       ind indl ind2)
  (dotimes (i ll)
	   (setq ind (floor (/ (- (aref dt i) xl) bw)))
	   (setf (aref bc ind) (1+ (aref bc ind)))
	   (setq indl (floor (/ (- (aref dt i) xl) (/ bw 100))))
	   (setf (aref lbc indl) (1+ (aref lbc indl)))
	   (setq ind2 (floor (/ (- (aref dt i) xl) (/ bw 2))))
	   (setf (aref 2bc ind2) (1+ (aref 2bc ind2))))
  (slot-value 'long-bin-counts lbc)
  (slot-value 'half-bin-counts 2bc)
  bc))

;;(defmeth-fde-proto-accessor-changes :boxplot-lines 
;;"not yet"
;;(compute-slot-value-default))


(defmeth-fde-proto-accessor-changes :hohisto-lines
"Lines for the hollow histogram"
(let* ((be (send self :bin-edges))
       (bc (* (send self :bin-counts)
	      (send self :y-factor)))
       xs ys)
  (labels ((add-point (x y)
		      (setf xs (cons x xs))
		      (setf ys (cons y ys))))
	  (add-point (select be 0) 0)
	  (dotimes (i (length bc))
		   (add-point (select be i) (select bc i))
		   (add-point (select be (1+ i)) (select bc i)))
	  (add-point (select be (1- (length be))) 0))
  (list (reverse xs)  (reverse ys))))

(defmeth-fde-proto-accessor-changes :piecelin-lines
"lines for the piecewise linear estimator, equal bins assumed"
; equal bins assumed
(let* ((edg (send self :bin-edges))
       (bc (* (send self :bin-counts)
	      (send self :y-factor)))
       (bc2 (/ (slot-value 'half-bin-counts)
	       (send self :bin-width) .5
	       (caadr (send self :data-summary))))
       (bc2 (* (slot-value 'half-bin-counts) 2 (send self :y-factor)))
       xs ys incy)
  (dotimes (i (length bc))
	   (setf xs (cons (select edg (list i (1+ i))) xs))
	   (setf incy (- (aref bc2 (1+ (* 2 i)))
			 (aref bc2 (* 2 i))))
	   (setf ys (cons (list (- (aref bc2 (* 2 i))
				   (/ incy 2))
				(+ (aref bc2 (1+ (* 2 i)))
				   (/ incy 2)))
			  ys)))
  (list (reverse xs) (reverse ys))))

(defmeth-fde-proto-accessor-changes :histo-lines
"lines for the regular histogram"
(let* ((be (send self :bin-edges))
       (bc (* (send self :bin-counts) (send self :y-factor)))
       xs ys)
  (labels ((add-point (x y)
		      (setf xs (cons x xs))
		      (setf ys (cons y ys))))
	  (dotimes (i (1- (length bc)))
		   (add-point (select be i) 0)
		   (add-point (select be i) (select bc i))
		   (add-point (select be (1+ i)) (select bc i))
		   (add-point (select be (1+ i)) 0)))
  (list (reverse xs)  (reverse ys)    )))

(defmeth-fde-proto-accessor-changes :fpoly-lines
"lines for the frequency polygon"
(list (- (cdr (send self :bin-edges))
	 (/ (send self :bin-width) 2))
      (* (send self :bin-counts) (send self :y-factor))))

(defmeth-fde-proto-accessor-changes :edgpoly-lines
"lines for the edge frequency polygon"
(let ((bc (coerce (* (send self :bin-counts) (send self :y-factor))
		  'list)))
  (list (send self :bin-edges)
	(/ (+ (cons 0 bc) (append bc '(0))) 2))))

(defmeth-fde-proto-accessor-changes :linpoly-lines
"lines for the linnearly binned frequency polygon"
'not-implemented)

(defmeth-fde-proto-accessor-changes :density-lines
"lines for a density function to be drawn if there is :density slot"
(if (slot-value 'density)
    (let (xvals lines)
      (labels 
       ((args (x)
	      (case (slot-value 'density-num-args)
		    (1 (list x))
		    (2 (list x (cadadr (send self :data-summary))))
		    (3 (list x 
			     (second (cadr (send self :data-summary)))
			     (third (cadr (send self :data-summary))))))))
       (setf xvals (apply #'rseq 
			  (append (send self :x-range) '(200))))
       (setf lines
	     (transpose
	      (mapcar #'(lambda (x)
			  (list x (apply (slot-value 'density) (args x))))
		      xvals)))))
  nil))

(defmeth-fde-proto-accessor-changes :boxplot-lines
"lines to draw a simple box-plot"
  (when (slot-value 'window)
	(let* ((w (send self :to-window))
	       (y-pos (* (/ (max (send self :bin-counts)) 2)
			 (send self :y-factor)))
	       (yhi y-pos)
	       (ylo (* .94 yhi))
	       (yme (/ (+ yhi ylo) 2))
	       (fn (fivnum (send self :data)))
	       (points
		(macrolet ((mkpoint (n y) `(list (nth ,n fn) ,y)))
			  (list (mkpoint 0 yme)
				(mkpoint 1 yme) (mkpoint 1 yhi)
				(mkpoint 2 yhi) (mkpoint 2 ylo) (mkpoint 2 yhi)
				(mkpoint 3 yhi) (mkpoint 3 yme)
				(mkpoint 4 yme)
				(mkpoint 3 yme) (mkpoint 3 ylo)
				(mkpoint 1 ylo) (mkpoint 1 yme)))))
	  (transpose points))))


(defmeth-fde-proto-accessor-changes-draw :what-to-show
"sets (from a user demand) the slot what-to-show
 or retrieves it to be shown in the window"
'(:histo-lines))



;;; some special accessors
;;; not following the same structure


(defmeth fde-proto :density (&optional (value nil vset) (numargs 1))
"set/retr the slot. If slot contain a function, its graph will be drawn. 
 The slot 'density-num-args must contain the number of args. 
 First is x, second is mean and 3rsd is scale (stdev)
 It is legal to send 'normal as value"
  (when vset
	;;set all dependants to nil
	(send self :mark-changed 'density)
	(if (eq value 'normal)
	    (progn
	      (slot-value 'density
			  #'(lambda (x m s)
			      (normal-dens (/ (- x m) s))))
	      (slot-value 'density-num-args 3))
	  (progn
	    (slot-value 'density value)
	    (slot-value 'density-num-args numargs))))
  (unless (slot-value 'density)
    (send self :density 'normal))
  (slot-value 'density))

(defmeth fde-proto :anchor-shift (&optional (value nil vset) &key (draw nil))
"this sets and/or returns a value in [0,1) that is the part of the binwidth 
 to be subtracted to the anchor-base to put the first bin edge there."
  (if vset
      (progn
	;;set all dependants to 'must-compute
	(send self :mark-changed 'anchor-shift)
	(loop
	 (when (< value 1) (return))
	 (setf value (1- value)))
	(loop
	 (when (>= value 0) (return))
	 (setf value (1+ value)))
	(slot-value 'anchor-shift value)
	(when draw (send self :to-window :redraw-content)))
    ;;not vset, compute if needed
    (when (eq 'must-compute (slot-value 'anchor-shift))
	      (case (send self :anchor-base)
		    ('data-based (slot-value 'anchor-shift 0.5))
		    ('zero (slot-value 'anchor-shift 0)))))
  (slot-value 'anchor-shift))

(defmeth fde-proto :bin-frequencies ()
"this uses bin-counts and rely in equal bin width"
(/ (send self :bin-counts)
   (send self :bin-width)
   (caadr (send self :data-summary))))

(defmeth fde-proto :stability-index ()
"computes the stability index for the histogram as defined in 
 Simonoff-Udina (1995). The index gives an idea of the stability of
 the histogram shape when the anchor position is changed. An index 
 of 1 is for a totally stable histogram, whose shape do not change
 at all when anchor is shifted.
 A list id returned: the stab-index for the histogram and stab-index
 for the edge freq polygon."
    ;;compute if needed
    (when (eq 'must-compute (slot-value 'stability-index))
	  (slot-value
	   'stability-index
	   (let* ((nb (length (send self :bin-counts)))
		  (k (1- nb))
		  (lbc (slot-value 'long-bin-counts))
		  (ap 100) ; number of anchor positions
		  (ll (* ap nb)) ; length of long-bin-counts
		  (mj (make-array (* k ap) :initial-element 0))
		  (dj (make-array (* (1- k) ap) :initial-element 0))
		  (st (make-array ap :initial-element 0));histo
		  (smft (make-array ap :initial-element 0));edge-freq poly
		  tmp tt i)
	     (dotimes
	      (i ll)
	      (cond ((< i ap); 0 <= i < ap
		     (setf (aref mj 0)
			   (+ (aref mj 0)
			      (aref lbc i))))
		    ((< i (+ ap ap)); ap <= i < 2 ap
		     (setf j (- i ap))
		     (setf (aref mj (1+ j))
			   (+ (aref mj j)
			      (aref lbc i)
			      (- (aref lbc j))))
		     (setf (aref smft (rem j ap))
			   (+ (aref smft (rem j ap))
			      (* (aref mj j) (aref mj j)))))
		    ((< i (- ll ap)); 2 ap <= i < ll - ap
		     (setf j (- i ap))
		     (setf jj (- i ap ap))
		     (setf (aref mj (1+ j))
			   (+ (aref mj j)
			      (aref lbc i)
			      (- (aref lbc j))))
		     (setf (aref smft (rem j ap))
			   (+ (aref smft (rem j ap))
			      (* (aref mj j) (aref mj j))))
		     (setf (aref dj jj)
			   (- (aref mj j) (aref mj jj)))
		     (setf (aref st (rem jj ap))
			   (+ (aref st (rem jj ap))
			      (* (aref dj jj) (aref dj jj)))))
		    (t ; ll > i > ll-ap
		     (setf j (- i ap))
		     (setf jj (- i ap ap))
		     (setf tt (- ap (- ll i)))
		     (when (< i (1- ll))
			      (setf (aref mj (1+ j))
				    (+ (aref mj j)
				       (- (aref lbc j)))))
		     (setf (aref smft (rem j ap))
			   (+ (aref smft (rem j ap))
			      (* (aref mj j) (aref mj j))))
		     (setf (aref dj jj)
			   (- (aref mj j) (aref mj jj)))
		     (setf (aref st (rem jj ap))
			   (+ (aref st (rem jj ap))
			      (* (aref dj jj) (aref dj jj))))
		     (setf (aref st tt)
			   (+ (aref st tt)
			      (* (aref mj tt) (aref mj tt))
			      (* (aref mj j) (aref mj j)))))))
;;	     (break)
	     (list
	      (gini-index (coerce st 'list))
	      (gini-index (coerce smft 'list)))
)))
    (slot-value 'stability-index))


(defmeth fde-proto :add-to-show (what &key (remove nil) (draw t))
"args: what &key (remove nil) (draw t)
WHAT is a list of (:histo-lines :hohisto-lines :fpoly-lines
:edgpoly-lines :linpoly-lines :piecelin-lines :data :boxplot-lines) keyword. 
These keywords are added to slot 'what-to-show unless already there
or removed if keyword :REMOVE is t"
(unless (listp what)
	(setf what (list what)))
(if remove
    (slot-value 'what-to-show
		(set-difference (slot-value 'what-to-show) what :test #'eq))
  (slot-value 'what-to-show
	      (union (slot-value 'what-to-show) what :test #'eq)))
(send self :window-up-to-date 'must-compute)
(when draw
      (send self :redraw-window)))

(defmeth fde-proto :window-up-to-date (&optional (value nil vset))
"called with 'must-compute to mark that window is not up-to-date
 called w/o args to enforce computations to be up-to-date.
 It returns t if it was already up-to-date, nil otherwise
"
  (if vset
      (progn
	;;set all dependants to changed
	(send self :mark-changed 'window-up-to-date)
	(slot-value 'window-up-to-date value))
    ;;not vset, compute if needed
    (if (eq 'must-compute (slot-value 'window-up-to-date))
	(progn
	  (mapcar #'(lambda (m)
		      (send self m))
		  (send self :what-to-show))
	  (slot-value 'window-up-to-date t)
	  nil)
      (slot-value 'window-up-to-date t))))

;;;
;;; some method to connect fde and its window
;;;
;;;
;;;

(defmeth fde-proto :have-window (&key (show t))
  (if (slot-value 'window)
      (send (slot-value 'window) :show-window)
    (progn
      (slot-value 'window
		  (send wfde-proto
			:new :show show :title (slot-value 'title)
			:fde-core self))
      (send self :window-up-to-date 'must-compute)
      (send self :to-window :adjust-to-data :draw t)))
  (slot-value 'window))

(defmeth fde-proto :show-info-in-window (&rest args)
)

(defmeth fde-proto :to-window (&rest args)
  (if (slot-value 'window)
      (if  args
	  (apply #'send (slot-value 'window) args)
	(slot-value 'window))))

(defmeth fde-proto :redraw-window ()
  (send self :to-window :redraw))

;;;
;;; Some computation functions and methods needed
;;; and other utilities,
;;; configuration vars, etc
;;;


(defmacro sqr (x) `(* ,x ,x))

(defmeth fde-proto :compute-default-bin-width (&key (type 'normal-reference))
  (ecase type 
	 ('normal-reference 
	  (* 3.49 (send self :scale-estimate)
	     (^ (caadr (send self :data-summary)) -0.333333333)))
	 ('sturgess-rule
	  (let* ((ds (send self :data-summary))
		 (xmi (first (first ds)))
		 (xma (fifth (first ds)))
		 (nn (first (second ds)))
		 (nbins (round (1+ (/ (log nn) (log 2))))))
	    (values (/ (* 1.01 (- xma xmi))
		       nbins) 
		    nbins)))
	 ))

(defmeth fde-proto :stability-index-values (&optional 
					    (from (first (send self :bw-ends)))
					    (to (second (send self :bw-ends)))
					    (num 50))
"returns a list of 1) bin-width values, 2) index for the histograms
 and 3) index for the edge frequency polygon."
  (let* ((bw0 (send self :bin-width))
	 (vals (rseq from to num))
	 (idxs (mapcar #'(lambda (bw)
			   (send self :bin-width bw :draw nil)
			   (send self :stability-index))
		       vals))
	 (tr (transpose idxs)))
    (send self :bin-width bw0)
    (list vals (first tr) (second tr))))


(defmeth fde-proto :assess-index-value (&key (stability-index
					      (first (send self :stability-index)))
					     (bin-counts
					      (send self :bin-counts))
					     (number-of-anchors 100)
					     (bootstrap-size 400)
					     (verbose t))
  (let* ((nn (sum bin-counts))
	 (d (length bin-counts))
	 (nless 0) (ntot 0)
	 shaps idxs pl perc)
    (labels
     ((one-multinomial-rand; thanks devroye and deleeuw
       (); no args
       (let* ((p (/ bin-counts nn))
	      (s 1)
	      (m nn)
	      (x (make-list d)))
	 (dotimes (k d x)
		  (let ((pp (elt p k)))
		    (cond ((= m 0) (setf (elt x k) 0))
			  ((>= pp s);just for the case that fp arit gives that
			   (decf m (setf (elt x k) m))
			   (decf s pp))
			  (t (decf m (setf (elt x k) 
					   (car (binomial-rand 1 m (/ pp s)))))
			     (decf s pp)))))))
      (one-index-value ()
		       (setf shaps nil)
		       (dotimes (i number-of-anchors (gini-index shaps))
				(setf shaps (cons
					     (sum (sqr (difference 
							(append '(0)
								(one-multinomial-rand)
								'(0)))))
					     shaps)))))
     ;;body begins here
     ;;(break)
     (when verbose 
	   (format t "Bin-width: ~a~%histogram index value to assess: ~a~%" 
		   (send self :bin-width) stability-index)
	   (format t "~a histograms will be simulated~%" bootstrap-size))
     (dotimes (b bootstrap-size)
	      (setf idx (one-index-value))
	      ;;(print (list idx stability-index))
	      (setf idxs (cons idx idxs))
	      (when (< idx stability-index)
		    (incf nless))
	      (when 
	       (and verbose 
		    (> b 1)
		    (= 19 (rem b 20)))
	       (format t
		       "~,4g  >=  ~,3g% simulated values (B=~a)~%"
		       stability-index
		       (/ (* 100 nless) (1+ b))
		       (1+ b))))
     (when verbose (format t
			   "So, the Monte Carlo evidence level for G=~,4g  ~% in this histogram is  ~,2g. (B=~a)~%"
		       stability-index
		       (/ nless bootstrap-size)
		       bootstrap-size)
	   (if (< (/ nless bootstrap-size) 0.05)
	       (format t " Reject the stability of the histogram,~% the unstability is due to the anchor position problem.~%")
	     (format t " The stability of the histogram can't be rejected~%"))
	     )
     (/ nless bootstrap-size))))


(defmeth fde-proto :num-bins ()
"the actual number of bins with non-zero counts"
  (let* ((bc (send self :bin-counts))
	 (nb (length bc))
	 (zf 0) (zl 0))
    (loop (unless (and (< zf nb) (= 0 (aref bc zf))) (return))
	  (setf zf (1+ zf)))
    (if (>= zf nb)
	0
      (loop (unless (= 0 (aref bc (- nb zl 1))) (return (- nb zf zl)))
	    (setf zl (1+ zl))))))

(defun gini-index (alist); this is the faster to compute
"Given ALIST of positive-or-zero number, gives a Gini-like index measuring
 the diversity of the numbers. Ranges in [0 1]."
  (let* ((nli (sort-data alist))
	 (nn (length nli))
	 (sim (sum (* (iseq nn 1) nli)))
	 (sm (sum nli)))
    (/ (1- (* 2 (/ sim sm))) nn)))

(defun featurep (sym)
  (member sym *features*))

(defun system-has-color ()
  (or (featurep 'color)
      (featurep :color)))

(unless (fboundp 'system-has-windows)
	(defun system-has-windows ()
	  (or (member 'windows *features*)
	      (member :windows *features*))))


;;  (if (featurep :macintosh) '(400 280)  '(650 450))
(defvar *kde-wsize*
(round (* 0.7 (screen-size)))
"the default size of windows to be created")

(defvar *line-colors* 
  (cond
   ((featurep :msdos) '(magenta blue yellow green red cyan white black))
   (t '(yellow green red cyan magenta blue white black))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; wfde is the object to display
;;; histogram and polygon frequency estimates
;;; is similar to wkde-proto
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defproto wfde-proto '(fde-core) () graph-proto)

(defmeth wfde-proto :to-core (&rest args)
  (if (null args)
      (slot-value 'fde-core)
      (apply #'send (slot-value 'fde-core) args)))

(defmeth wfde-proto :isnew (&key (title nil) fde-core 
				(show nil) (debug nil))
  (call-next-method 2 :show nil :title title 
	 :menu-template '(mouse dash rescale options to-gnuplot
				dash info stability-index show-index-plot 
				assess-index
				dash 
				bin-control y-scale what-to-show
				animate)
	 :menu-title "Fde")
  (setf (slot-value 'fde-core) fde-core)

  (send self :add-mouse-mode 'show-coordinates 
	:title "Show Coordinates" :cursor 'finger 
	:click :do-show-coordinates)
  (send self :add-mouse-mode 'zoom-in 
	:title "Zoom in" :cursor 'cross 
	:click :do-zoom-in)
  (send self :delete-mouse-mode 'selecting)
  (send self :delete-mouse-mode 'brushing)
  (send self :delete-mouse-mode 'point-moving)
  (send self :mouse-mode 'show-coordinates)

;;  (send self :margin 0 0 0 (* 3 (send self :text-ascent)))
  (send self :x-axis t t 5)
  (send self :y-axis t t 4)
  (send self :adjust-to-data :draw nil)
  (send self :update nil)
  
  (apply #'send self :size (/ (screen-size) 2));;*kde-wsize*
  (when (featurep :unix)
	  (send self :back-color 'black)
	  (send self :draw-color 'white))
  (when show (send self :show-window)
	(send self :redraw-content)
	(send self :to-core :show-info-in-window)
	(send self :activate t);;in release 3 this is needed???
	)
  self
)

(defmeth wfde-proto :draw-data ()
"draw a cross for each data point in top of window"
(let* ((sym 'dot)
       (w self)
       (ym (second (send w :scaled-range 1)))
       (dt (sort-data (send self :to-core :data))) 
       xyloc (lastxyloc '(-99999 -99999)))
    (dolist (pnt dt)
         (setf xyloc (send w :real-to-canvas pnt ym))
         (when (= (first xyloc) (first lastxyloc))
               (setf (second xyloc) (+ 1 (second lastxyloc))))
         (apply #'send w :draw-symbol sym nil
		     xyloc)
         (setf lastxyloc xyloc))))



(defmeth wfde-proto :redraw-content ()
  (let* ((core (send self :to-core))
	 (wts (send core :what-to-show))
	 (clip (send self :clip-rect))
	 (denslines (send core :density-lines)))
    (unless
     (send core :window-up-to-date)
     (send self :clear-lines :draw nil)
     (loop
      (unless wts (return))
      (case (first wts)
	    (:histo-lines
	     (send self :add-lines (send core :histo-lines) 
		   :draw nil :color (select *line-colors* 0)))
	    (:hohisto-lines
	     (send self :add-lines (send core :hohisto-lines) 
		   :draw nil :color (select *line-colors* 0)))
	    (:fpoly-lines
	     (send self :add-lines (send core :fpoly-lines) 
		   :draw nil :color (select *line-colors* 1)))
	    (:edgpoly-lines
	     (send self :add-lines (send core :edgpoly-lines) 
		   :draw nil :color (select *line-colors* 2)))
	    (:piecelin-lines
	     (mapcar #'(lambda (li) 
			 (send self :add-lines  li
			       :draw nil :color (select *line-colors* 3)))
		     (transpose (send core :piecelin-lines))))
	    (:boxplot-lines
	     (send self :add-lines (send core :boxplot-lines) 
		   :draw nil :color (select *line-colors* 4)))
	    (:density-lines
	     (send self :add-lines denslines
		   :draw nil :color (select *line-colors* 4)))
	    (:linpoly-lines
	     (print "linearly binned poly not implemented yet"))
	    )
      (setf wts (cdr wts)))
;      (when denslines
; 	   (send self :add-lines denslines
; 		 :draw nil :color (select *line-colors* 4)))
     )
    (apply #'send self :clip-rect (send self :content-rect))
    (call-next-method)
    (apply #'send self :clip-rect clip)
    (when (member :data (send self :to-core :what-to-show))
	  (send self :draw-data))
    ;;    (apply #'send self :paint-rect (send self :content-rect) )
    ;;  (send self :draw-info-strings)
    ))

(defmeth wfde-proto :adjust-to-data (&key (draw t))
  (let ((core (send self :to-core))
	rng)
    (send self :redraw-content)
    (call-next-method :draw nil)
    (setf rng (send self :range 1))
    ;;(if (< (abs (first rng)) 0.0001)
    (setf (first rng) 0)
    (send self :range 1 (first rng) (second rng) :draw nil)
    (setf rng (send core :x-range))
    (send self :range 0 (first rng) (second rng) :draw nil)
    (send self :redraw)))	


(defmeth wfde-proto :close ()
"Before closing its windows, closes other child windows"
(call-next-method)
(send self :to-core :slot-value 'window nil)
;(send self :to-core :close-windows)
)


(defun define-show-coordinates (obj)
  (defmeth obj :do-show-coordinates (x y m1 m2)
    ;;modified from graphics.lsp, xlispstat for mac
    ;;this version allows moving the mouse while seeing the coordinates
    ;;of the click point
    (let* ((xy (cond (m1 (list x y))
		     (m2 (send self :canvas-to-scaled x y))
		     (t (send self :canvas-to-real x y))
		     ))
	   (s (format nil "(~,3g, ~,3g)" (first xy) (second xy)))
	   (str-size (send self :text-width s))
	   (left (> (+ x str-size) (send self :canvas-width)))
         (horz (if left 2 0))
         (vert 0))
      (send self :draw-string-while-button s x y horz vert)))
  (defmeth obj :draw-string-while-button (s x y h v)
    (let* ((oldx x)
	   (oldy y)
	   (origin (first (transpose (send self :scaled-range '(0 1)))))
	   (origin (apply #'send self :scaled-to-canvas origin))
	   (mode (send self :draw-mode)))
      (send self :draw-mode 'xor)
      (send self :draw-line x y (first origin) y)
      (send self :draw-line x y x (second origin))
      (send self :draw-text s x y h v)
      (send self :while-button-down
	    #'(lambda (nx ny)
		(send self :draw-text s oldx oldy h v)
		(send self :draw-text s nx ny h v)
		(setq oldx nx oldy ny)))
					;redraw things for erasing
      (send self :draw-text s oldx oldy h v)
      (send self :draw-line x y (first origin) y)
      (send self :draw-line x y x (second origin))
      (send self :draw-mode mode)))
  )

(define-show-coordinates wfde-proto)

(defmeth wfde-proto :do-zoom-in (x y m1 m2)
(if (or m1 m2)
  (send self :adjust-to-data)
  (let* ((oldx x)
         (oldy y)
         (old-draw-mode (send self :draw-mode))
         (rect (progn (send self 
                            :while-button-down
                            #'(lambda (nx ny)
                                (send self :draw-mode 'xor)
                                (send self :frame-rect x y
                                      (- oldx x) (- oldy y))
                                (send self :frame-rect x y
                                      (- nx x) (- ny y))
                                (setf oldx nx)
                                (setf oldy ny)))
                      (list x y (- oldx x) (- oldy y))))
         (l (nth 0 rect))
         (to (nth 1 rect))
         (w (nth 2 rect))
         (h (nth 3 rect))
         (lb (send self :canvas-to-scaled l (+ to h)))
         (tr (send self :canvas-to-scaled (+ l w) to))
         (xmin (nth 0 lb))
         (xmax (nth 0 tr))
         (ymin (nth 1 lb))
         (ymax (nth 1 tr)))
    (send self :frame-rect l to w h)
    (send self :draw-mode old-draw-mode)
    (send self :set-ranges (min xmin xmax) (max xmin xmax)
          (min ymin ymax) (max ymin ymax)))))


(defmeth wfde-proto :set-ranges (xmin xmax ymin ymax &key (recalc nil))
  (send self :scaled-range 0 xmin xmax :draw nil)
  (send self :scaled-range 1 ymin ymax :draw nil)
  (when recalc (error "set-ranges does not recalculate" nil)) ;this is to be done...
#-macintosh
  (send self :redraw)
  (send self :update nil))

(defmeth wfde-proto :locate-dialog (dialog)
#-msdos"will return a good position for a dialog to appear"
#+msdos"this method does nothing in MS-Windows"
;it tries to locate first in rigth
;then in down
;and if all fails, over left-up corner of myself
#+msdos()
#-msdos(let* ((mypos (send self :location))
       (dlgsize (if (send dialog :size)
		    (send dialog :size)
		  '(150 150)))
       (screen (screen-size))
       (mysize (send self :size))
       
       (all (- screen (+ mypos mysize dlgsize))))
  (cond ((> (first all) 0)
	 (apply #'send dialog :location (list (+ (first mypos)
						 (first mysize))
					      (second mypos))))
	((> (second all) 0)
	 (apply #'send dialog :location (list (first mypos)
					      (+ (second mypos)
						 (second mysize)))))
	(t (apply #'send dialog :location (+ '(10 20) mypos))))))


(defmeth wfde-proto :make-menu-item (item-template)
  (let ((it (call-next-method item-template)))
    (if it it
      (case item-template
	    (to-gnuplot (require "gnuplot"
				 (merge-pathnames (make-pathname :name "gnuplot") 
						  *fde-directory*))
			(send graph-item-proto :new "Graph to gnuplot file" self
			      :to-gnuplot))
	    (animate (send graph-item-proto :new "Animate anchor moving" self 
               :toggle-animation :idle-on :toggle t))
	    (show-index-plot (send graph-item-proto :new "Stability index plot..." 
				   self :show-index-plot))
	    (info (send graph-item-proto :new "Show info..." 
				   self :info-dialog))
	    (stability-index (send graph-item-proto :new "Stability index..." self 
               :show-stability-index))
	    (assess-index (send graph-item-proto :new "Assess the index value..."
				   self 
				   :assess-index-value))
	    (bin-control (send graph-item-proto :new "Bin control..." self 
               :pop-bins-dialog))
	    (y-scale (send graph-item-proto :new "Vertical scale..." self 
               :y-scale-dialog))
	    (what-to-show (send graph-item-proto :new "What to show..." self 
               :what-to-show-dialog))
)))) 

;;;
;;; interface methods
;;;


(defmeth wfde-proto :do-idle ()
  (send self :to-core :anchor-shift
	(+ 0.04 (send self :to-core :anchor-shift)) :draw t))

(defmeth wfde-proto :toggle-animation (&optional (bool nil set))
  (if set (send self :idle-on bool)
    (send self :idle-on (not (send self :idle-on)))))

(defmeth wfde-proto :show-stability-index ()
  (let ((core (Send self :to-core))
	dlg str1 str2 gind)
    (unless (listp (send core :slot-value 'stability-index))
	    (setf str1 (format nil
			       "Please, wait for the index~%to be computed"))
	    (setf dlg
		  (send dialog-proto :new
			(list (send text-item-proto :new str1))
			:show t
			:title "Computing the index"))
	    (send dlg :show-window))
    (setf gind (send self :to-core :stability-index))
    (when dlg (send dlg :dispose))
    (message-dialog
     (princ
      (format nil
	      "For bin width: ~,3g~%stability index for the histogram~%is G = ~,3g~%For the edge frequency polygon~%is G = ~,3g~%"
	      (send self :to-core :bin-width)
	      (first gind) (second gind))))))

(defmeth wfde-proto :show-index-plot ()
  (let* ((core (Send self :to-core))
	 (bwe (send core :bw-ends))
	 (str (format nil "~,4g ~,4g ~a" 
		      (first bwe) (second bwe) 50))
	 (resp (get-string-dialog
		(format nil "This will compute and display stability index values~%for a series of bin width values in a range.~%It is a some minute(s) computation.~%Enter bin width range:~%    from  to  num-values")
		:initial str))
	 pl lins win)
    (when resp
	  (setq win
		(send dialog-proto :new
		      (list (send text-item-proto
				  :new "This will take some time..."))
		      :title "This will take some time..."))
	  (send win :show-window)
	  (labels ((evalstring (aString)
		    "Returns a list of the expressions obtained 
on evaluation of the characters in aString"  
		    (let ((st (make-string-input-stream aString))   
			  (result nil)   
			  (expr nil))  
		      (loop (setq expr (read st nil 'eof))  
			    (when (eql expr 'eof)    
				  (return (reverse result)))  
			    (setf result (cons expr result))))))
		  (setq lins (apply #'send core :stability-index-values
				    (evalstring resp)))
		  (require "my-plot-lines" 
			   (merge-pathnames (make-pathname :name "plotline" :type "") 
					    *fde-directory*))
                  (send win :dispose)
		  (if (find "my-plot-lines" *modules* :test #'string=)
		      (setq pl
			    (my-plot-lines (first lins)
					   (cdr lins)))
		    (progn
		      (setq pl (plot-lines (first lins) (second lins)))
		      (send pl :add-lines (list (first lins) (third lins)))
		      (define-show-coordinates pl)
		      (send pl :add-mouse-mode 'show-coordinates 
			    :title "Show Coordinates" :cursor 'finger 
			    :click :do-show-coordinates)
		      (send pl :mouse-mode 'show-coordinates)
		      ))
		  lins))))

(defmeth wfde-proto :info-dialog ()
  (let* ((cor (send self :to-core))
	 (ds (send cor :data-summary))
	 (str (format
	       nil
	       "~a~,3g~%~a~a~%~a~,3g ~,3g~%~a~,3g ~,3g ~,3g~%~a~a~%~a~,3g~%~a~,3g"
	       "Current bin-width: " (send cor :bin-width)
	       "Number of bins: " (send cor :num-bins)
	       "Data range: " (first (first ds)) (fifth (first ds))
	       "Quartiles: " (second (first ds)) (third (first ds)) (fourth
								     (first ds))
	       "N         = " (first (second ds))
	       "Mean      = " (second (second ds))
	       "Std. dev. = " (third (second ds)))))
    (message-dialog str :title "Current data set")
    (list (send cor :bin-width) (send cor :num-bins)
	  (first (first ds)) (fifth (first ds))
	  (second (first ds)) (third (first ds))(fourth (first ds))
	  (first (second ds))(second (second ds))(third (second ds)))))

(defmeth wfde-proto :pop-bins-dialog ()
  (let* ((core (send self :to-core))
	 (bwpr (send text-item-proto :new
		     (format nil "Bin-width:~% ")))
	 (bwdp (send text-item-proto :new "" :text-length 6))
	 (bwsl (send interval-scroll-item-proto :new
		       (send core :bw-ends)
		       :text-item bwdp
		       :action #'(lambda(x) (send core :bin-width x :draw t))))
	 (aspr (send text-item-proto :new
		     (format nil "Anchor shift~%(% of the bin)")))
	 (asdp (send text-item-proto :new "" :text-length 6))
	 (assl (send interval-scroll-item-proto :new
		       '(0 100)
		       :text-item asdp
		       :action #'(lambda(x) (send core :anchor-shift (- 1 (/ x 100))
						  :draw t))
		       :points 21))
	 (items (list (list (list bwsl (list bwpr bwdp) )
			    (list assl (list aspr asdp) ))))
	 (dlg (send dialog-proto :new items :title "Bin control" :show nil)))
    (send dlg :location
	  (first (send self :location))
	  (+ 10 (second (send self :location))
	     (second (send self :size))))
    (send dlg :show-window)
    (send bwsl :value (send core :bin-width))
    (send assl :value (send core :anchor-shift))
    (send self :add-subordinate dlg)))

(defmeth wfde-proto :y-scale-dialog ()
"Pops a dialog to choose the vertical scale for the histogram"
(let* ((core (send self :to-core))
       (opt '("Density scale" "Frequency scale" "Count scale"))
       (syms '(density frequency count))
       (resp (choose-item-dialog "Choose the vertical scale"
				 opt
				 :initial (position (send core :y-scale) syms))))
  (when resp (send core :y-scale (nth resp syms) :draw t))))
  

(defmeth wfde-proto :what-to-show-dialog ()
  (let* ((core (send self :to-core))
	 (sh '(:histo-lines :hohisto-lines :fpoly-lines
               :edgpoly-lines :piecelin-lines :data :boxplot-lines :density-lines));:linpoly-lines 
	 (str '("Histogram" "Hollow histogram" "Frequency polygon"
		"Edge frequency polygon" 
		"Piecewise linear estimator" "Data points" "Simple boxplot" "A (normal) density shadow"));"Linearly binned freq. poly"
	 (in (which (mapcar #'(lambda (x)
				(find x (send core :what-to-show)))
			    sh)))
	 (resp (choose-subset-dialog "Mark what must be shown:"
				     str :initial in)))
    (when (and resp (caar resp))
	  (send core :what-to-show (select sh (car resp)) :draw t))))

(defmeth wfde-proto :assess-index-value ()
  (message-dialog 
   (format nil "Simulation will run~% in the listener window"))
  (send self :to-core :assess-index-value :bootstrap-size 200 :number-of-anchors 50))

(defmeth wfde-proto :do-key (char shift option)
"some keyboard interface"
(case char
      (#\+
       (send self :to-core
	     :bin-width (* 1.2 (send self :to-core :bin-width)) :draw t)
       (send self :to-core :show-info-in-window nil "" ""))
      (#\-
       (send self :to-core
	     :bin-width (/ (send self :to-core :bin-width) 1.2) :draw t)
       (send self :to-core :show-info-in-window nil "" ""))
      (#\a 
       (send self :adjust-to-data))
      ((#\r) (send self :to-core :anchor-shift
		   (- (send self :to-core :anchor-shift) 0.1) :draw t))
      ((#\l) (send self :to-core :anchor-shift
		   (+ (send self :to-core :anchor-shift) 0.1) :draw t))
      (#\d  (send self :to-core :add-to-show '(:data)
		  :remove (member :data (send self :to-core 
					      :slot-value 'what-to-show))))
      ))

;;;
;;; Some data sets analyzed in Simonoff-Udina(1995)
;;;

;; Old faithful geyser eruptions duration in August 1978 and August 1979.
(defvar oldf-dur '(4.4 3.9 4 4 3.5 4.1 2.3 4.7 1.7 4.9 1.7 4.6 3.4 4.3 1.7 3.9 3.7 3.1 4 1.8 4.1 1.8 3.2 1.9 4.6 2 4.5 3.9 4.3 2.3 3.8 1.9 4.6 1.8 4.7 1.8 4.6 1.9 3.5 4 3.7 3.7 4.3 3.6 3.8 3.8 3.8 2.5 4.5 4.1 3.7 3.8 3.4 4 2.3 4.4 4.1 4.3 3.3 2 4.3 2.9 4.6 1.9 3.6 3.7 3.7 1.8 4.6 3.5 4 3.7 1.7 4.6 1.7 4 1.8 4.4 1.9 4.6 2.9 3.5 2 4.3 1.8 4.1 1.8 4.7 4.2 3.9 4.3 1.8 4.5 2 4.2 4.4 4.1 4.1 4 4.1 2.7 4.6 1.9 4.5 2 4.8 4.1 4.1 4.2 4.5 1.9 4.7 2 4.7 2.5 4.3 4.4 4.4 4.3 4.6 2.1 4.8 4.1 4 4 4.4 4.1 4.3 4 3.9 3.2 4.5 2.2 4.7 4.6 2.2 4.8 4.3 3.8 4 4.1 1.8 4.4 4 2.2 5.1 1.9 5 4.4 4.5 3.8 4.3 4.4 2.2 4.8 1.9 4.7 1.8 4.8 2 4.4 2.5 4.3 4.4 1.9 4.7 4.3 2.2 4.7 2.3 4.6 3.3 4.2 2.9 4.6 3.3 4.2 2.6 4.6 3.7 1.8 4.7 4.5 4.5 4.8 2 4.8 1.9 4.7 2 5.1 4.3 4.8 3 2.1 4.6 4 2.2 5.1 2.9 4.3 2.1 4.7 4.5 1.7 4.2 4.3 1.7 4.4 4.2 2.2 4.7 4 1.8 4.7 1.8 4.5 2.1 4.2 2.1 5.2 2))

(defvar pcb-concentrations '(77.55 29.23 403.1 736 192.15 220.6 8.62 174.31 529.28 130.67 39.74 0.0 8.43 0.0 120.04 0.0 11.93 0.0 0.0 30.14 0.0 0.0 0.0 531.67 9.3 5.74 46.47 176.9 13.69 4.89 6.6 6.73 4.28 20.5 20.5 5.8 5.08 ))

(defvar buffalo-snowfall '(126.4 82.4 78.1 51.1 90.9 76.2 104.5 87.4 110.5 25 69.3 53.5 39.8 63.6 46.7 72.9 79.6 83.6 80.7 60.3 79 74.4 49.6 54.7 71.8 49.1 103.9 51.6 82.4 83.6 77.8 79.3 89.6 85.5 58 120.7 110.5 65.4 39.9 40.1 88.7 71.4 83 55.9 89.9 84.8 105.2 113.7 124.7 114.5 115.6 102.4 101.4 89.8 71.5 70.9 98.3 55.5 66.1 78.4 120.5 97 110))

(defvar visa-data '( 1.612800 6.021000E-01 1.707600 2.250400 1.770900 1.079200 2.420000 1.792400 2.721800 1.740400 1.699000 1.041400 2.086400 6.990000E-01 2.510500 1.716000 2.387400 1.602100 1.397900 2.651300 1.591100 1.919100 1.230400 2.025300 1.041400 1.204100 9.542000E-01 1 2.248000 2.858500 2.620100 1.977700 1 3.406900 3.259400 1.740400 2.103800 7.782000E-01 1.230400 ))


(defvar oldf-pauses '(78 74 68 76 80 84 50 93 55 76 58 74 75 80 56 80 69 57 90 42 91 51 79 53 82 51 76 82 84 53 86 51 85 45 88 51 80 49 82 75 73 67 68 86 72 75 75 66 84 70 79 60 86 71 67 81 76 83 76 55 73 56 83 57 71 72 77 55 75 73 70 83 50 95 51 82 54 83 51 80 78 81 53 89 44 78 61 73 75 73 76 55 86 48 77 73 70 88 75 83 61 78 61 81 51 80 79 82 80 76 56 82 47 76 61 75 72 74 69 78 52 91 66 71 75 81 77 74 70 83 53 82 62 73 84 58 82 77 75 77 77 53 75 78 51 81 52 76 73 84 72 89 75 57 81 49 87 43 94 45 81 59 82 80 54 75 73 57 80 51 77 66 77 60 86 62 75 67 69 84 58 90 82 71 80 51 80 62 84 51 81 83 84 72 54 75 74 51 91 60 80 54 80 70 60 86 78 51 83 76 51 90 71 49 88 52 79 61 81 48 84 63 ))

(setq nba-ages '(28 30 26 30 28 31 30 27 29 24 27 29 24 30 28 32 25 29 34 23 32 28 28 23 32 27 34 26 30 30 23 31 28 27 25 32 29 34 28 23 26 30 32 27 27 25 24 27 25 27 31 30 25 26 33 24 26 31 24 27 28 22 30 31 23 25 31 33 28 37 28 24 34 24 28 33 23 26 28 26 25 25 26 25 27 35 31 25 30 24 23 23 27 27 25 24 24 23 23 26 24 23 32 24 27))


(defun fde-demo ()
  (setf fd (send fde-proto :new :data (coerce oldf-dur 'vector) :x-range '(0 6)
		 :title "fde: Geiser data")))


(defun stability-index (data bin-width &optional max-bin-width number-of-values)
"Args: data bin-width &optional max-bin-width number-of-values
 Computes the Simonoff-Udina stability index for DATA and BIN-WIDTH. If given optional
 arguments max-bin-width number-of-values, the index is computed for a list of binwidths
 from bin-width to max-bin-width."
  (let ((fde (send fde-proto :new :data (coerce data 'vector) :bin-width bin-width :show nil))
	bws res)
    (if (and max-bin-width number-of-values)
	(progn
	  (setq bws (rseq bin-width max-bin-width number-of-values))
	  (list bws 
		(mapcar #'(lambda (b)
			    (send fde :bin-width b)
			    (send fde :stability-index))
			bws)))
      (send fde :stability-index))))

;;;provide fde
(provide "fde")
